// Copyright 2014 Google Inc. All Rights Reserved.

#include "NotificationEndpoint.h"

void NotificationEndpoint::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    // Magic proto-hack to make it show up even though it is empty.
    service->mutable_generic_notification_service();
}

int NotificationEndpoint::routeMessage(uint8_t channelId, uint16_t type,
        const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    switch (type) {
        case GENERIC_NOTIFICATION_SUBSCRIBE: {
            // GenericNotificationSubscribe is empty. Don't bother parsing.
            mSubscribed = true;
            mCallbacks->subscriptionStatusCallback(true);
            ret = STATUS_SUCCESS;
            break;
        }
        case GENERIC_NOTIFICATION_UNSUBSCRIBE: {
            // GenericNotificationUnsubscribe is empty. Don't bother parsing.
            mSubscribed = false;
            mCallbacks->subscriptionStatusCallback(false);
            ret = STATUS_SUCCESS;
            break;
        }
        case GENERIC_NOTIFICATION_MESSAGE: {
            GenericNotificationMessage notif;
            if (PARSE_PROTO(notif, ptr, len)) {
                uint8_t* icon = NULL;
                size_t iconLen = 0;
                if (notif.has_icon()) {
                    icon = (uint8_t*) notif.icon().c_str();
                    len = notif.icon().size();
                }
                mCallbacks->notificationCallback(notif.text(), notif.has_id(), notif.id(),
                        notif.has_icon(), icon, iconLen);
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case GENERIC_NOTIFICATION_ACK: {
            GenericNotificationAck ack;
            if (PARSE_PROTO(ack, ptr, len)) {
                mCallbacks->ackCallback(ack.id());
                ret = STATUS_SUCCESS;
            }
            break;
        }
    }

    return ret;
}

int NotificationEndpoint::sendNotification(const string& text, bool hasId, const string& id,
        bool hasIcon, uint8_t* icon, size_t len) {
    if (!mSubscribed) {
        return STATUS_INVALID_CHANNEL;
    }
    GenericNotificationMessage msg;
    msg.set_text(text);
    if (hasId) {
        msg.set_id(id);
    }
    if (hasIcon) {
        msg.set_icon(icon, len);
    }
    IoBuffer buf;
    mRouter->marshallProto(GENERIC_NOTIFICATION_MESSAGE, msg, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int NotificationEndpoint::ackNotification(const string& id) {
    if (!mSubscribed) {
        return STATUS_INVALID_CHANNEL;
    }
    GenericNotificationAck msg;
    msg.set_id(id);
    IoBuffer buf;
    mRouter->marshallProto(GENERIC_NOTIFICATION_ACK, msg, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}
